<?php

namespace src\router;

use Exception;
use src\controllers\HomeController; // Ajusta la ruta del controlador según tu estructura

/**
 * Clase Route para manejar el enrutamiento de la aplicación.
 */
class Route
{
    /**
     * @var string Módulo obtenido de la URL.
     */
    private $module;

    /**
     * @var string Método del controlador obtenido de la URL.
     */
    private $method;

    /**
     * @var string Parámetro opcional obtenido de la URL.
     */
    private $parameter;

    /**
     * Constructor de la clase Route.
     */
    public function __construct()
    {
        $method = strtolower($_SERVER['REQUEST_METHOD']);
        $url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '/';
        $urlParts = explode('/', ltrim($url, '/'));
        $this->module = isset($urlParts[0]) && !empty($urlParts[0]) ? $urlParts[0] : 'home';
        $this->method = $method . (isset($urlParts[1]) && !empty($urlParts[1]) ? ucfirst($urlParts[1]) : ucfirst($this->module));
        $this->parameter = isset($urlParts[2]) ? $urlParts[2] : null;
    }

    /**
     * Despacha la solicitud, instanciando el controlador y llamando al método correspondiente.
     * @return mixed Resultado del método del controlador.
     */
    public function dispatch()
    {
        $controllerClass = ucfirst($this->module) . 'Controller';
        $controllerInstance = $this->instantiateController($controllerClass);

        try {
            return $this->controllerMethod($controllerInstance);
        } catch (Exception $e) {
            return ('Error: ' . $e->getMessage());
        }
    }

    /**
     * Instancia el controlador correspondiente.
     * @param string $controllerClass Nombre de la clase del controlador.
     * @return object Instancia del controlador.
     * @throws Exception Cuando no se encuentra el controlador.
     */
    private function instantiateController($controllerClass)
    {
        $fullControllerClass = "src\\controllers\\$controllerClass";
        if (class_exists($fullControllerClass)) {
            return new $fullControllerClass();
        } else {
            $this->send400Response("Controlador no encontrado");
        }
    }

    /**
     * Llama al método del controlador y devuelve el resultado.
     * @param object $controllerInstance Instancia del controlador.
     * @return mixed Resultado del método del controlador.
     * @throws Exception Cuando la acción no es válida.
     */
    private function controllerMethod($controllerInstance)
    {
        if (method_exists($controllerInstance, $this->method)) {
            $filteredParameter = $this->sanitizeParameter($this->parameter);
            $result = $controllerInstance->{$this->method}($filteredParameter);
            return $this->minifyHTML($result);
        } else {
            $this->send400Response("Método no válido");
        }
    }
    /**
     * Minifica el HTML si no estamos en un entorno de desarrollo.
     * @param string $html HTML a minificar.
     * @return string HTML minificado o sin cambios.
     */
    private function minifyHTML($html)
    {

        // if (CONFIG_APP['envDev']) {
        //     $minifiedHtml = preg_replace('/>\s+</', '><', $html);
        //     $html = trim($minifiedHtml);
        // }
        return $html;
    }
    /**
     * Limpia un parámetro permitiendo solo números y letras.
     * @param string $parameter Parámetro a limpiar.
     * @return string Parámetro limpio.
     */
    private function sanitizeParameter($parameter)
    {
        // Utilizar una expresión regular para permitir solo números y letras
        return preg_replace('/[^a-zA-Z0-9]/', '-', $parameter, -1);
    }

    /**
     * Envía una respuesta HTTP 400 y muestra un mensaje.
     * @param string $message Mensaje a mostrar.
     */
    private function send400Response($message)
    {
        header("HTTP/1.1 400 Bad Request");
        echo $message;
        exit;
    }
}
